// 🚀 **Helper Functions (Now shared by both vanilla JS & jQuery)**
function getCookie(name) {
  const cookies = document.cookie.split("; ");
  for (let i = 0; i < cookies.length; i++) {
    let [cookieName, cookieValue] = cookies[i].split("=");
    if (cookieName === name) {
      try {
        const decodedValue = decodeURIComponent(cookieValue); // ✅ First decode
        return JSON.parse(decodedValue); // ✅ Then parse JSON
      } catch (e) {
        console.error("⚠ Cookie parse error:", e);
        return null;
      }
    }
  }
  return null;
}

function setCookie(name, value, days) {
  let expires = "";
  if (days) {
    const date = new Date();
    date.setTime(date.getTime() + days * 24 * 60 * 60 * 1000);
    expires = "; expires=" + date.toUTCString();
  }
  const cookieString =
    name +
    "=" +
    encodeURIComponent(JSON.stringify(value)) + // ✅ Proper encoding
    expires +
    "; path=/; SameSite=Lax";

  document.cookie = cookieString;
}

function sendConsentToGoogle(consentStatus) {
  window.dataLayer = window.dataLayer || [];
  window.dataLayer.push({
    event: "default_consent",
    analytics_storage: consentStatus ? "granted" : "denied",
    ad_storage: consentStatus ? "granted" : "denied",
    personalization_storage: consentStatus ? "granted" : "denied",
  });

  gtag("consent", "update", {
    analytics_storage: consentStatus ? "granted" : "denied",
    ad_storage: consentStatus ? "granted" : "denied",
    personalization_storage: consentStatus ? "granted" : "denied",
  });
}

// 🚀 **Vanilla JavaScript Implementation**
document.addEventListener("DOMContentLoaded", function () {
  const modal = document.getElementById("cookie-preferences-modal");
  const acceptBtn = document.getElementById("accept-all-cookies");
  const rejectBtn = document.getElementById("reject-all-cookies");

  // ✅ Check if user already set preferences
  const cookieData = getCookie("cookie_preferences");

  if (!cookieData || cookieData.all === false) {
    if (!cookieData) console.log("→ No cookie found!");
    else if (cookieData.all === false) console.log("→ User rejected cookies!");
    modal.style.display = "flex"; // Show modal if no consent
  } else {
    modal.style.display = "none"; // Hide modal if consent is already set
  }

  acceptBtn.addEventListener("click", function () {
    setCookie("cookie_preferences", { essential: true, all: true }, 365);
    sendConsentToGoogle(true);
    modal.style.display = "none";
  });

  rejectBtn.addEventListener("click", function () {
    setCookie("cookie_preferences", { essential: true, all: false }, 365);
    sendConsentToGoogle(false);
    modal.style.display = "none";
  });
});

// 🚀 **jQuery Implementation**
jQuery(document).ready(function ($) {
  const modal = $("#cookie-preferences-modal");
  const acceptBtn = $("#accept-all-cookies");
  const rejectBtn = $("#reject-all-cookies");

  // ✅ Check if user already set preferences
  const cookieData = getCookie("cookie_preferences");

  if (!cookieData || !cookieData.all) {
    modal.fadeIn(); // Use jQuery animation for smooth display
  }

  acceptBtn.on("click", function () {
    setCookie("cookie_preferences", { essential: true, all: true }, 365);
    sendConsentToGoogle(true);
    modal.fadeOut(); // Hide the modal smoothly
  });

  rejectBtn.on("click", function () {
    setCookie("cookie_preferences", { essential: true, all: false }, 365);
    sendConsentToGoogle(false);
    modal.fadeOut();
  });
});
